<?php
/**
 * Minimal Attendance Webhook
 * Receives attendance data from Hikvision device
 */

header('Content-Type: application/json');
date_default_timezone_set('Asia/Kathmandu');

// Data file path
$logFile = __DIR__ . '/attendance.json';

// Initialize if not exists
if (!file_exists($logFile)) {
    file_put_contents($logFile, json_encode([]));
}

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['status' => 405, 'message' => 'Method not allowed']));
}

// Get event log
$eventLog = $_POST['event_log'] ?? '';
if (empty($eventLog)) {
    http_response_code(400);
    die(json_encode(['status' => 400, 'message' => 'No event_log data']));
}

// Parse JSON
$data = json_decode($eventLog, true);
if (!$data) {
    http_response_code(400);
    die(json_encode(['status' => 400, 'message' => 'Invalid JSON']));
}

// Extract data
$event = $data['AccessControllerEvent'] ?? [];
$employeeId = $event['employeeNoString'] ?? '';
$eventType = $event['subEventType'] ?? 0;
$eventTime = $data['dateTime'] ?? '';
$deviceIp = $data['ipAddress'] ?? '';

// Validate
if (empty($employeeId) || $event['majorEventType'] != 5) {
    http_response_code(400);
    die(json_encode(['status' => 400, 'message' => 'Invalid event']));
}

// Only accept today's events
$eventDate = date('Y-m-d', strtotime($eventTime));
if ($eventDate !== date('Y-m-d')) {
    http_response_code(400);
    die(json_encode(['status' => 400, 'message' => 'Expired event']));
}

// Determine type
$type = 'staff';
if (preg_match('/^stu/i', $employeeId)) {
    $hour = (int)date('H');
    $type = $hour < 12 ? 'student-in' : 'student-out';
}

// Create record
$record = [
    'id' => uniqid(),
    'employee_id' => $employeeId,
    'type' => $type,
    'event_type' => $eventType,
    'event_time' => $eventTime,
    'device_ip' => $deviceIp,
    'recorded_at' => date('Y-m-d H:i:s')
];

// Load existing logs
$logs = json_decode(file_get_contents($logFile), true) ?? [];
$logs[] = $record;

// Keep last 5000 records
if (count($logs) > 5000) {
    $logs = array_slice($logs, -5000);
}

// Save
file_put_contents($logFile, json_encode($logs, JSON_PRETTY_PRINT));

// Response
echo json_encode([
    'status' => 200,
    'message' => 'Success',
    'data' => ['id' => $record['id'], 'type' => $type, 'time' => date('H:i:s', strtotime($eventTime))]
]);
