<?php
/**
 * Simple Attendance Dashboard
 * View attendance logs from Hikvision device
 */

date_default_timezone_set('Asia/Kathmandu');

$logFile = __DIR__ . '/attendance.json';
$logs = file_exists($logFile) ? json_decode(file_get_contents($logFile), true) : [];
$today = date('Y-m-d');

// Filter by type
$filter = $_GET['type'] ?? 'all';
$filtered = $filter === 'all' ? $logs : array_filter($logs, fn($l) => $l['type'] === $filter);

// Get today's logs
$todayLogs = array_filter($logs, fn($l) => strpos($l['event_time'], $today) === 0);
$stats = [
    'total' => count($todayLogs),
    'student-in' => count(array_filter($todayLogs, fn($l) => $l['type'] === 'student-in')),
    'student-out' => count(array_filter($todayLogs, fn($l) => $l['type'] === 'student-out')),
    'staff' => count(array_filter($todayLogs, fn($l) => $l['type'] === 'staff'))
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Attendance System</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        header { background: #2c3e50; color: white; padding: 20px; text-align: center; margin-bottom: 20px; }
        .stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px; }
        .stat-card { background: white; padding: 20px; border-radius: 8px; text-align: center; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }
        .stat-card h3 { color: #7f8c8d; font-size: 14px; margin-bottom: 10px; }
        .stat-card .number { font-size: 32px; font-weight: bold; color: #2c3e50; }
        .filters { background: white; padding: 15px; border-radius: 8px; margin-bottom: 20px; display: flex; gap: 10px; flex-wrap: wrap; }
        .filters a { padding: 10px 20px; background: #ecf0f1; color: #2c3e50; text-decoration: none; border-radius: 5px; }
        .filters a.active { background: #3498db; color: white; }
        table { width: 100%; background: white; border-collapse: collapse; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }
        th { background: #34495e; color: white; padding: 12px; text-align: left; }
        td { padding: 12px; border-bottom: 1px solid #ecf0f1; }
        tr:hover { background: #f8f9fa; }
        .badge { padding: 5px 10px; border-radius: 12px; font-size: 12px; font-weight: 600; }
        .badge.in { background: #d4edda; color: #155724; }
        .badge.out { background: #fff3cd; color: #856404; }
        .badge.staff { background: #d1ecf1; color: #0c5460; }
        .empty { text-align: center; padding: 60px 20px; color: #95a5a6; }
        .info-box { background: #e8f4f8; padding: 15px; border-left: 4px solid #3498db; border-radius: 5px; margin-top: 20px; }
    </style>
</head>
<body>
    <header>
        <h1>📋 Attendance System</h1>
        <p>Hikvision Device Integration</p>
    </header>

    <div class="container">
        <div class="stats">
            <div class="stat-card">
                <h3>Total Today</h3>
                <div class="number"><?= $stats['total'] ?></div>
            </div>
            <div class="stat-card">
                <h3>Student In</h3>
                <div class="number"><?= $stats['student-in'] ?></div>
            </div>
            <div class="stat-card">
                <h3>Student Out</h3>
                <div class="number"><?= $stats['student-out'] ?></div>
            </div>
            <div class="stat-card">
                <h3>Staff</h3>
                <div class="number"><?= $stats['staff'] ?></div>
            </div>
        </div>

        <div class="filters">
            <a href="?" class="<?= $filter === 'all' ? 'active' : '' ?>">All</a>
            <a href="?type=student-in" class="<?= $filter === 'student-in' ? 'active' : '' ?>">Student In</a>
            <a href="?type=student-out" class="<?= $filter === 'student-out' ? 'active' : '' ?>">Student Out</a>
            <a href="?type=staff" class="<?= $filter === 'staff' ? 'active' : '' ?>">Staff</a>
            <button onclick="location.reload()" style="margin-left: auto; padding: 10px 20px; background: #27ae60; color: white; border: none; border-radius: 5px; cursor: pointer;">🔄 Refresh</button>
        </div>

        <?php if (empty($filtered)): ?>
            <div class="empty">
                <h2>No Records Found</h2>
                <p>Attendance records will appear here when employees tap the device</p>
            </div>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Date/Time</th>
                        <th>Employee ID</th>
                        <th>Type</th>
                        <th>Device IP</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $count = 1;
                    foreach (array_reverse($filtered) as $log): 
                        $badgeClass = $log['type'] === 'staff' ? 'staff' : ($log['type'] === 'student-in' ? 'in' : 'out');
                    ?>
                        <tr>
                            <td><?= $count++ ?></td>
                            <td><?= date('M j, Y H:i:s', strtotime($log['event_time'])) ?></td>
                            <td><strong><?= htmlspecialchars($log['employee_id']) ?></strong></td>
                            <td><span class="badge <?= $badgeClass ?>"><?= strtoupper(str_replace('-', ' ', $log['type'])) ?></span></td>
                            <td><?= htmlspecialchars($log['device_ip']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>

        <div class="info-box">
            <strong>📡 Webhook Endpoint:</strong> 
            <code style="background: white; padding: 5px 10px; border-radius: 3px; margin-left: 10px;">
                https://dev.ntranepal.edu.np/attendance100/webhook.php
            </code>
            <p style="margin-top: 10px; font-size: 14px;">Configure this URL in your Hikvision device settings</p>
        </div>
    </div>

    <script>
        // Auto-refresh every 30 seconds
        setTimeout(() => location.reload(), 30000);
    </script>
</body>
</html>
